#!/usr/bin/perl
=pod

=head1 NAME

map.pl -- Display map

=head1 SYNOPSIS

perl map.pl

=head1 DESCRIPTION

This program displays both topographical maps and areal photographs
for a given location.  The maps can be panned and zoomed.

Maps are downloaded and assembled from tiles downloaded from
USGS data and the terraserver web site.  See 
L<http://teraserver.homeadvisor.msn.com/About/AbuotLinktoHtml.htm>
for more information.

The GUI should be self explanatory.  

=head1 RESTRICTIONS

The tile directory is hard coded to a UNIX style directory
(I<~/.maps>).   It will need to be changed for windows use.

=head1 BUGS

The program never clears any files out of the tile cache.

=cut



use strict;
use warnings;

# TODO:
#
# Click - Zoom
#
#	Allow %d in a file name
#
#	Keyboard h,j,k,l
#	Q / < > (,. as well)
#	p -- print
# 
#	

use Tk;
use Geo::Coordinates::UTM;
use HTTP::Lite;
use Tk::Photo;
use Tk::JPEG;
use Tk::LabEntry;
use Tk::BrowseEntry;
use Image::Magick;

use constant MAP_PHOTO => 1;	# Areal Photograph
use constant MAP_TOPO => 2;	# Topo map 

my $tk_mw;		# Main window
my $tk_canvas;		# Canvas on the main window
my $tk_nav;		# Navigation window

my $x_size = 3;		# Size of the map in X
my $y_size = 3;		# Size of the map in Y
my $map_type = MAP_TOPO;# Type of the map
my $scale = 12;		# Scale for the map

my $goto_long = 0;	# Where to go from the entry
my $goto_lat = 0;

my @tk_scale_buttons;	# The buttons to display the scale


my $center_lat;
my $center_long;

my $cache_dir = "$ENV{HOME}/.maps";
if (! -d $cache_dir) {
    if (not mkdir($cache_dir, 0755)) {
	die("Could not create cache directory");
    }
}

########################################################
# do_error -- Display an error dialog
########################################################
sub do_error($)
{
    my $msg = shift;	# Error message to display
    $tk_mw->messageBox(
	-title => "Error",
	-message => $msg,
	-type => "OK",
	-icon => "error"
    );
}
    
########################################################
# convert_fract($) -- Convert to factional degrees
#
#	Knows the formats:
#		dddmmss
#		dd.ffff		(not converted)
########################################################
sub convert_fract($)
{
    my $value = shift;	# Value to convert

    # Fix the case where we have things 
    # like 12345W or 13456S
    if ($value =~ /^([+-]?\d+)([nNeEsSwW])$/) {
	my $code;	# Direction code
	($value, $code) = ($1, $2);
	if (($code eq 's') || ($code eq 'S') || 
	    ($code eq 'W') || ($code eq 'w')) {
	    $value = -$value;
	}
    }
    # Is it a long series of digits with possible sign?
    if ($value =~ /^[-+]?\d+$/) {
	# USGS likes to squish things to together +DDDmmSS
	# Get the pieces
	$value =~ /([-+]?)(\d+)(\d\d)(\d\d)/;
	my ($sign, $deg, $min, $sec) = ($1, $2, $3, $4);
	
	# Convert to fraction
	my $result = ($deg + ($min / 60.0) + 
	             ($sec / (60.0*60.0)));

	# Take care of sign
	if ($sign eq "-") {
	    return (-$result);
	}
	return($result);
    }
    if ($value =~ /^[-+]?\d*\.\d*$/) {
	return ($value);
    }
    print "Unknown format for ($value)\n";
    return (undef);
}
#######################################################
# set_center_lat_long($lat, $long) -- 
#	Change the center of a picture
########################################################
sub set_center_lat_long($$)
{
    my $lat = shift;	# Coordinate of the map	(latitude)
    my $long = shift;	# Coordinate of the map (longitude)

    $lat = convert_fract($lat);
    $long = convert_fract($long);

    if (defined($long) and defined($lat)) {
	$center_lat = $lat;
	$center_long = $long;
    }
}

#
# Scales from 
#	http://teraserver.homeadvisor.msn.com/
#		/About/AbuotLinktoHtml.htm
#
# Fields
#	Resolution -- Resolution of the 
#			map in meter per pixel
#	factor -- Scale factor to turn UTM into tile number
#	doq -- Aerial photo available
#	drg -- Topo map available
#
my %scale_info = (
    10=>{resolution=>  1, factor=>   200, doq=>1, drg=>0},
    11=>{resolution=>  2, factor=>   400, doq=>1, drg=>1},
    12=>{resolution=>  4, factor=>   800, doq=>1, drg=>1},
    13=>{resolution=>  8, factor=>  1600, doq=>1, drg=>1},
    14=>{resolution=> 16, factor=>  3200, doq=>1, drg=>1},
    15=>{resolution=> 32, factor=>  6400, doq=>1, drg=>1},
    16=>{resolution=> 64, factor=> 12800, doq=>1, drg=>1},
    17=>{resolution=>128, factor=> 25600, doq=>0, drg=>1},
    18=>{resolution=>256, factor=> 51200, doq=>0, drg=>1},
    19=>{resolution=>512, factor=>102400, doq=>0, drg=>1}
);
########################################################
# map_to_tiles()
#
# Turn a map into a set of URLs
#
# Returns the url array
########################################################
sub map_to_tiles()
{
    my @result;

    # Get the coordinates as UTM
    my ($zone,$easting,$north)=latlon_to_utm(
	'GRS 1980',$center_lat, $center_long);

    # Fix the zone, it must be a number
    $zone =~ /(\d+)/;
    $zone = $1;

    # Compute the center tile number
    my $center_x = 
        int($easting / $scale_info{$scale}->{factor});

    my $center_y = 
        int($north / $scale_info{$scale}->{factor});

    # Compute the starting location
    my $start_x = $center_x - int($x_size / 2);
    my $start_y = $center_y - int($y_size / 2);

    # Compute the ending location
    my $end_x = $start_x + $x_size;
    my $end_y = $start_y + $y_size;

    for (my $y = $end_y-1; $y >= $start_y; --$y) {
	for (my $x = $start_x; $x < $end_x; ++$x) {
	    push (@result, { 
				T => $map_type,  
				S => $scale, 
				X => $x,
				Y => $y,
				Z =>$zone}
	    );
	}
    }
    return (@result);
}

########################################################
# get_file($) -- Get a photo file from an URL
#
########################################################
sub get_file($)
{
    my $url = shift;	# URL to get

    my $file_spec = 
       "$cache_dir/t=$url->{T}_s=$url->{S}_x=$url->{X}_y=$url->{Y}_z=$url->{Z}.jpg";
    if (! -f $file_spec) {
	# Connection to the remote site
	my $http = new HTTP::Lite;

	# The image to get
	my $image_url = "http://terraserver-usa.com/tile.ashx?T=$url->{T}&S=$url->{S}&X=$url->{X}&Y=$url->{Y}&Z=$url->{Z}";
	print "Getting $image_url\n";

	# The request
	my $req = $http->request($image_url);
	if (not defined($req)) {
	    die("Could not get url $image_url");
	}

	# Dump the data into a file
	my $data = $http->body();
	open (OUT_FILE, ">$file_spec") or 
	   die("Could not create $file_spec");
	print OUT_FILE $data;
	close OUT_FILE;
    }
    return ($file_spec);
}
########################################################
# Given a state name, return the 
#	file with the information in it
########################################################
sub info_file($)
{
    my $state = shift;	# State we have

    # The file we need for this state
    my $file_spec = "$cache_dir/${state}_info.txt";
    return ($file_spec);
}
########################################################
# get_place_file($) -- 
#	Get a place information file for the give state
########################################################
sub get_place_file($)
{
    my $state = shift;	# URL to get

    # The file we need for this state
    my $file_spec = info_file($state);

    if (! -f $file_spec) {
	# Connection to the remote site
	my $http = new HTTP::Lite;

	# The image to get
	my $place_url = "http://geonames.usgs.gov/stategaz/${state}_DECI.TXT";
	print "Getting $place_url\n";

	# The request
	my $req = $http->request($place_url);
	if (not defined($req)) {
	    die("Could not get url $place_url");
	}

	# Dump the data into a file
	my $data = $http->body();
	open (OUT_FILE, ">$file_spec") or 
	   die("Could not create $file_spec");
	print OUT_FILE $data;
	close OUT_FILE;
    }
    return ($file_spec);
}
########################################################
# get_photo($) -- Get a photo from a URL
########################################################
sub get_photo($)
{
    my $url = shift;	# Url to get

    # File containing the data
    my $file_spec = get_file($url);

    my $tk_photo = $tk_mw->Photo(-file => $file_spec);
    return ($tk_photo);
}

########################################################
# paint_map(@maps)
#
# Paint a bitmap on the canvas
########################################################
sub paint_map(@)
{
    my @maps = @_;	# List of maps to display

    # Delete all the old map items
    $tk_canvas->delete("map");

    for (my $y = 0; $y < $y_size; ++$y) {
	for (my $x = 0; $x < $x_size; ++$x) {
	    my $url = shift @maps;	# Get the URL 
	    # Turn it into a photo
	    my $photo = get_photo($url);
	    $tk_canvas->createImage($x * 200, $y * 200,
		-tags => "map",
		-anchor => "nw",
		-image => $photo);
	}
    }
    $tk_canvas->configure(
	-scrollregion => [ $tk_canvas->bbox("all")]);
}

########################################################
# show_map -- Show the current map
########################################################
sub show_map()
{
    my @result = map_to_tiles();
    # Repaint the screen
    paint_map(@result);
}
########################################################
# do_move($x, $y) -- Move the map in the X and Y direction
########################################################
sub do_move($$)
{
    my $x = shift;
    my $y = shift;

    my ($zone,$east,$north)=
        latlon_to_utm('GRS 1980',$center_lat, $center_long);

    $east -= $x * $scale_info{$scale}->{factor};
    $north -= $y * $scale_info{$scale}->{factor};

    ($center_lat, $center_long) = 
        utm_to_latlon('GRS 1980', $zone, $east, $north);
    show_map();
}
########################################################
# change_type -- Toggle the type of the map
########################################################
sub change_type() {
    if ($map_type == MAP_TOPO) {
	if ($scale_info{$scale}->{doq}) {
	    $map_type = MAP_PHOTO;
	}
    } else {
	if ($scale_info{$scale}->{drg}) {
	    $map_type = MAP_TOPO;
	}
    }
    set_scale($scale);
    show_map()
}
########################################################
# set_scale($new_scale) -- Change the scale to a new value
########################################################
sub set_scale($) {
    my $new_scale = shift;	# The scale we want to have

    if (not defined($scale_info{$new_scale})) {
	return;
    }
    if ($map_type == MAP_TOPO) {
	if (not $scale_info{$new_scale}->{drg}) {
	    return;
	}
    } else {
	if (not $scale_info{$new_scale}->{doq}) {
	    return;
	}
    }
    $scale = $new_scale;
    for (my $i = 0; $i <= $#tk_scale_buttons; ++$i) {
	if (($i + 10) == $scale) {
	    $tk_scale_buttons[$i]->configure(
		-background => "green"
	    );
	} else {
	    # The background
	    my $bg = "white";
	    if ($map_type == MAP_TOPO) {
		if (not $scale_info{$i+10}->{drg}) {
		    $bg = "gray";
		}
	    } else {
		if (not $scale_info{$i+10}->{doq}) {
		    $bg = "Gray";
		}
	    }
	    $tk_scale_buttons[$i]->configure(
		-background => $bg
	    );
	}
    }
    show_map();
}
########################################################
# change_canvas_size -- Change the size of the canvas
########################################################
sub change_canvas_size()
{
    if ($x_size <= 0) {
	$x_size = 1;
    }
    if ($y_size <= 0) {
	$y_size = 1;
    }
    $tk_canvas->configure(
	-width => $x_size * 200, -height => $y_size * 200);
    show_map();
}
my $save_image_name = "map_image"; # The name of the image file to save
my $tk_save_image;	# The save image popup

use Image::Magick;
########################################################
# do_save_image -- 
#	Save the image as a file (actually do the work)
########################################################
sub do_save_image()
{
    if ($save_image_name !~ /\.(jpg|jpeg)$/) {
	$save_image_name .= ".jpg";
    }

    # List of tiles to write
    my @tiles = map_to_tiles();

    # The image array
    my $images = Image::Magick->new();

    # Load up the image array
    foreach my $cur_tile (@tiles) {
	# The file containing the tile
	my $file = get_file($cur_tile);

	# The result of the read
	my $result = $images->Read($file);
	if ($result) {
	    print "ERROR: for $file -- $result\n";
	}
    }

    # Put them together
    my $new_image = $images->Montage(geometry => "200x200",
    	tile => "${x_size}x$y_size");

    my $real_save_image_name = $save_image_name;
    if ($save_image_name =~ /%d/) {
	for (my $i = 0; ; ++$i) {
	    $real_save_image_name = 
	        sprintf($save_image_name, $i);
	    if (! -f $real_save_image_name) {
		last;
	    }
	}
    }
    # Save them
    $new_image->Write($real_save_image_name);
    $tk_save_image->withdraw();
    $tk_save_image = undef;
}

########################################################
# save_image -- Display the save image popup
########################################################
sub save_image()
{
    if (defined($tk_save_image)) {
	$tk_save_image->deiconify();
	$tk_save_image->raise();
	return;
    }
    $tk_save_image = $tk_mw->Toplevel(
	-title => "Save Image");

    $tk_save_image->LabEntry(
	-label => "Name: ", 
	-labelPack => [ -side => 'left'],
	-textvariable => \$save_image_name
    )->pack(
	-side => "top",
	-expand => 1,
	-fill => 'x'
    );
    $tk_save_image->Button(
	-text => "Save",
	-command => \&do_save_image
    )->pack(
	-side => 'left'
    );
    $tk_save_image->Button(
	-text => "Cancel",
	-command => sub {$tk_save_image->withdraw();}
    )->pack(
	-side => 'left'
    );
}
########################################################
# print_image -- Print the image to the default printer
#	(Actually save it as postscript)
########################################################
sub print_image()
{
    # List of tiles to write
    my @tiles = map_to_tiles();

    # The image array
    my $images = Image::Magick->new();

    # Load up the image array
    foreach my $cur_tile (@tiles) {
	# The file containing the tile
	my $file = get_file($cur_tile);

	# The result of the read
	my $result = $images->Read($file);
	if ($result) {
	    print "ERROR: for $file -- $result\n";
	}
    }

    # Put them together
    my $new_image = $images->Montage(geometry => "200x200",
    	tile => "${x_size}x$y_size");

    my $print_file;	# File name for printing

    for (my $i = 0; ; ++$i) {
	if (! -f "map.$i.ps") {
	    $print_file = "map.$i.ps";
	    last;
	}
    }
    # Save them
    $new_image->Set(page => "Letter");
    $new_image->Write($print_file);
    $tk_mw->messageBox(
	-title => "Print Complete",
	-message => 
	    "Print Done.  Output file is $print_file",
	-type => "OK",
	-icon => "info"
    );
}
########################################################
# goto_lat_long -- Goto the given location
########################################################
sub goto_lat_long()
{
    set_center_lat_long($goto_lat, $goto_long);
}

my $tk_goto_loc;	# Goto location popup window
my $place_name;		# Name of the place to go to
my $state;		# State containing the place name

#
# The scrolling lists of data
#
# Fields
#   name --  The title of the data
#   index -- Index into the data fields for the place data
#   width -- Width of the field
#
my @data_list = (
    { 				# 0
	name => "Name",
	index => 2,
	width => 30
    },
    { 				# 1
	name => "Type",
	index => 3,
	width => 10,
    },
    {				# 2
	name => "County",
	index => 4,
	width => 20,
    },
    { 				# 3
	name => "Latitude",
	index => 7,
	width => 10,
    },
    { 				# 4
	name => "Longitude",
	index => 8,
	width => 10,
    },
    { 				# 5
	name => "Elevation",
	index => 15,
	width => 9,
    }
);

########################################################
# jump_to_loc -- 
#	Jump to the location specified in the list box
########################################################
sub jump_to_loc()
{
    my $cur_selection = 
        $data_list[0]->{tk_list}->curselection();

    if (not defined($cur_selection)) {
	do_error("You need to select an item to jump to");
	return;
    }
    # Where we're jumping to
    my $lat = 
       $data_list[3]->{tk_list}->get($cur_selection->[0]);

    my $long = 
        $data_list[4]->{tk_list}->get($cur_selection->[0]);

    set_center_lat_long($lat, $long);
    show_map();
}
########################################################
# scroll_listboxes -- Scroll all the list boxes
#	(taken from the O'Reilly book 
#	with little modification)
########################################################
sub scroll_listboxes
{
    my ($sb, $scrolled, $lbs, @args) = @_;

    $sb->set(@args);
    my ($top, $bottom) = $scrolled->yview();
    foreach my $list (@$lbs) {
	$list->{tk_list}->yviewMoveto($top);
    }
}
########################################################
# select_boxes -- Called when a Listbox gets a selection
#
#	So make everybody walk in lock step
########################################################
sub select_boxes($)
{
    # The widget in which someone selected
    my $tk_widget = shift;	

    my $selected = $tk_widget->curselection();

    foreach my $cur_data (@data_list) {
	$cur_data->{tk_list}->selectionClear(0, 'end');
	$cur_data->{tk_list}->selectionSet($selected->[0]);
    }
}

my $tk_place_where;	# The window with the places in it
########################################################
# do_goto_loc -- Goto a given location
########################################################
sub do_goto_loc()
{
    if ((not defined($state)) || ($state eq "")) {
	do_error("No state selected");
	return;
    }
    if (not defined($place_name)) {
	do_error("No place name entered");
	return;
    }
    if ($place_name =~ /^\s*$/) {
	do_error("No place name entered");
	return;
    }

    # The state as two character names
    my $state2 = substr($state, 0, 2); 
    get_place_file($state2);

    # The file containing the state information
    my $state_file = info_file($state2);

    open IN_FILE, "<$state_file" or 
        die("Could not open $state_file");

    my @file_data = <IN_FILE>;
    chomp(@file_data);
    close(IN_FILE);

    #TODO: Check to see if anything matched, if not error

    if (defined($tk_place_where)) {
	$tk_place_where->deiconify();
	$tk_place_where->raise();
    } else {
	# The pick a place screen
	$tk_place_where = $tk_mw->Toplevel(
	      -title => "Goto Selection");

	# Frame in which we place our places
	my $tk_place_frame = $tk_place_where->Frame();

	# The scrollbar for the place list
	my $tk_place_scroll = $tk_place_where->Scrollbar(
	)->pack(
	    -side => 'left', 
	    -fill => 'y'
	);

	# Loop through each item and construct it
	foreach my $cur_data (@data_list) {
	    $cur_data->{tk_frame} = 
	        $tk_place_frame->Frame();

	    $cur_data->{tk_frame}->Label(
		-text => $cur_data->{name}
	    )->pack(
		-side => 'top'
	    );
	    $cur_data->{tk_list} = 
	    	$cur_data->{tk_frame}->Listbox(
		-width => $cur_data->{width},
		-selectmode => 'single',
		-exportselection => 0
	    )->pack(
		-side => "top",
		-expand => 1,
		-fill => "both"
	    );
	    $cur_data->{tk_list}->bind(
		"<<ListboxSelect>>", \&select_boxes);

	    $cur_data->{tk_frame}->pack(-side => "left");
	    # Define how things scroll
	    $cur_data->{tk_list}->configure(
		-yscrollcommand => 
		    [ \&scroll_listboxes, 
		    $tk_place_scroll, 
		    $cur_data->{tk_list}, \@data_list]);
	}

	# define how the scroll bar works
	$tk_place_scroll->configure(
	    -command => sub {
		foreach my $list (@data_list) {
		    $list->{tk_list}->yview(@_);
		}
	    }
	);
	# Put the frame containing the list on the screen
	$tk_place_frame->pack(
	    -side => 'top', 
	    -fill => 'both', 
	    -expand => 1);

	$tk_place_where->Button(
	    -text => "Go To",
	    -command => \&jump_to_loc
	)->pack(
	    -side => 'left'
	);
	$tk_place_where->Button(
	    -text => "Close",
	    -command => sub { $tk_place_where->withdraw(); }
	)->pack(
	    -side => 'left'
	);
    }

    foreach my $cur_result (@file_data) {
	# Split the data up into fields
	# See http://gnis.usgs.gov for field list
	my @data = split /\|/, $cur_result;
	if ($data[2] !~ /$place_name/i) {
	    next;
	}
	foreach my $cur_data (@data_list) {
	    $cur_data->{tk_list}->insert('end', 
	    	$data[$cur_data->{index}]);
	}
    }
    foreach my $cur_data (@data_list) {
	$cur_data->{tk_list}->selectionSet(0);
    }
}


# List of states and two character abbreviations
my @state_list = (
    "AK = Alaska",
    "AL = Alabama",
    "AR = Arkansas",
    "AS = American Samoa",
    "AZ = Arizona",
    "CA = California",
    "CO = Colorado",
    "CT = Connecticut",
    "DC = District of Columbia",
    "DE = Delaware",
    "FL = Florida",
    "FM = Federated States of Micronesia",
    "GA = Georgia",
    "GU = Guam",
    "HI = Hawaii",
    "IA = Iowa",
    "ID = Idaho",
    "IL = Illinois",
    "IN = Indiana",
    "IT = All Indian Tribes",
    "KS = Kansas",
    "KY = Kentucky",
    "LA = Louisiana",
    "MA = Massachusetts",
    "MD = Maryland",
    "ME = Maine",
    "MH = Marshall Island",
    "MI = Michigan",
    "MN = Minnesota",
    "MO = Missouri",
    "MP = Northern Mariana Islands",
    "MS = Mississippi",
    "MT = Montana",
    "NC = North Carolina",
    "ND = North Dakota",
    "NE = Nebraska",
    "NH = New Hampshire",
    "NJ = New Jersey",
    "NM = New Mexico",
    "NV = Nevada",
    "NY = New York",
    "OH = Ohio",
    "OK = Oklahoma",
    "OR = Oregon",
    "PA = Pennsylvania",
    "PR = Puerto Rico",
    "PW = Palau, Republic of",
    "RI = Rhode Island",
    "SC = South Carolina",
    "SD = South Dakota",
    "TN = Tennessee",
    "TX = Texas",
    "UT = Utah",
    "VA = Virginia",
    "VI = Virgin Islands",
    "VT = Vermont",
    "WA = Washington",
    "WI = Wisconsin",
    "WV = West Virginia",
    "WY = Wyoming"
);
########################################################
# goto_loc -- Goto a named location 
#	(popup the window to ask the name)
########################################################
sub goto_loc()
{
    if (defined($tk_goto_loc)) {
	$tk_goto_loc->deiconify();
	$tk_goto_loc->raise();
	return;
    }
    $tk_goto_loc = $tk_mw->Toplevel(
	-title => "Goto Location");

    #TODO: Add label
    $tk_goto_loc->BrowseEntry(
	-variable => \$state,
	-choices => \@state_list,
    )->pack(
	-side => "top",
    );

    #TODO: Add place type
    $tk_goto_loc->LabEntry(
	-label => "Place Name: ", 
	-labelPack => [ -side => 'left'],
	-textvariable => \$place_name
    )->pack(
	-side => "top",
	-expand => 1,
	-fill => 'x'
    );
    $tk_goto_loc->Button(
	-text => "Locate",
	-command => \&do_goto_loc
    )->pack(
	-side => 'left'
    );
    $tk_goto_loc->Button(
	-text => "Cancel",
	-command => sub {$tk_goto_loc->withdraw();}
    )->pack(
	-side => 'left'
    );
}

# Mapping from direction to image names
my %images = (
    ul => undef,
    u => undef,
    ur => undef,
    l => undef,
    r => undef,
    dl => undef,
    d => undef,
    dr => undef,
);

my @key_bindings = (
    { key => "<Key-j>", event => sub { do_move(0, +1)}},
    { key => "<Key-k>", event => sub { do_move(0, -1)}},
    { key => "<Key-h>", event => sub { do_move(+1, 0)}},
    { key => "<Key-l>", event => sub { do_move(-1, 0)}},
    { key => "<Key-p>", event => \&print_image},
    { key => "<Key-q>", event => sub { exit(0)}},
    { key => "<Key-x>", event => sub { exit(0)}},
    { key => "<Key-s>", event => \&save_image},
);

########################################################
# build_gui -- Create all the GUI elements
########################################################
sub build_gui()
{
    $tk_mw = MainWindow->new(-title => "Topological Map");

    my $tk_scrolled = $tk_mw->Scrolled(
	'Canvas',
	-scrollbars => "sw"
    )->pack(
	-fill => "both",
	-expand => 1,
	-anchor => 'n',
	-side => 'top'
    );

    $tk_canvas = $tk_scrolled->Subwidget('canvas');
    $tk_canvas->configure(
	-height => 600,
	-width => 600
    );
    $tk_canvas->CanvasBind("<Button-1>", 
    	sub {set_scale($scale-1)});

    $tk_canvas->CanvasBind("<Button-2>", 
    	sub {set_scale($scale+1)});

    $tk_canvas->CanvasBind("<Button-3>", 
    	sub {set_scale($scale+1)});

    foreach my $cur_image (keys %images) {
	# The file to put in the image
	my $file_name = "arrow_$cur_image.jpg";

	# Create the image
	$images{$cur_image} = $tk_mw->Photo(
	    -file => $file_name);
    }
    $tk_mw->Button(-image => $images{ul}, 
	-command => sub {do_move(-1, 1)} )->grid(
	    $tk_mw->Button(
		-image => $images{u}, 
		-command => sub {do_move(0, 1)} 
	    ), 
	    $tk_mw->Button(
		-image => $images{ur}, 
		-command => sub {do_move(1, 1)}
	    ),
    	-sticky => "nesw"
    );
    $tk_mw->Button(-image => $images{l}, 
        -command => sub {do_move(-1, 0)} )->grid(
	    $tk_scrolled,
	    $tk_mw->Button(
		-image => $images{r}, 
		-command => sub {do_move(1, 0)}
	    ),
    	-sticky => "nesw"
    );
    $tk_mw->Button(
	-image => $images{dl}, 
	-command => sub {do_move(-1, -1)} 
    )->grid(
	$tk_mw->Button(
	    -image => $images{d}, 
	    -command => sub {do_move(0, -1)} 
	),
	$tk_mw->Button(
	    -image => $images{dr}, 
	    -command => sub {do_move(1, -1)} 
	),
    	-sticky => "nesw"
    );
    $tk_mw->gridColumnconfigure(1, -weight => 1);
    $tk_mw->gridRowconfigure(1, -weight => 1);

    # TODO: Is there some way of making this on top?
    $tk_nav = $tk_mw->Toplevel(-title => "Map Control");

    # Map the keys 
    foreach my $bind (@key_bindings) {
	$tk_mw->bind($bind->{key}, $bind->{event});
	$tk_nav->bind($bind->{key}, $bind->{event});
    }

    # The item to set the scale
    my $tk_scale_frame = $tk_nav->Frame();
    $tk_scale_frame->pack(-side => 'top', -anchor => 'w');

    $tk_scale_frame->Button(
	    -text => "+", 
	    -command => sub {set_scale($scale-1)}
	)->pack(
	    -side => 'right'
	);

    # Go through each scale and produce a button for it.
    foreach my $info (sort {$a <=> $b} keys %scale_info) {
	push(@tk_scale_buttons, $tk_scale_frame->Button(
	    -bitmap => "transparent",
	    -width => 10,
	    -height => 20,
	    -command => sub {set_scale($info);}
	)->pack(
	    -side => 'right'
	));
    }

    $tk_scale_frame->Button(
	-text => "-", 
	-command => sub {set_scale($scale+1) }
    )->pack(
	-side => 'right'
    );

    $tk_nav->Button(
	-text => "Toggle Type",
	-command => \&change_type
    )->pack(
	-side => "top",
	-anchor => "w"
    );


    # The frame for the X size adjustment
    my $tk_map_x = $tk_nav->Frame()->pack(
	    -side => "top", 
	    -fill => "x", 
	    -expand => 1
	);

    $tk_map_x->Label(
	    -text => "Map Width"
	)->pack(
	    -side => "left"
	);

    $tk_map_x->Button(
	    -text => "+", 
	    -command => sub {
		$x_size++, change_canvas_size()
	    }
	)->pack(
	    -side => "left"
	);
    $tk_map_x->Button(
	    -text => "-", 
	    -command => sub {
		$x_size--, change_canvas_size()
	    }
	)->pack(
	    -side => "left"
	);

    # The frame for the Y size adjustment
    my $tk_map_y = $tk_nav->Frame()->pack(
	-side => "top", 
	-fill => "x", 
	-expand => 1
    );
    $tk_map_y->Label(
	-text => "Map Height"
    )->pack(
	-side => "left"
    );
    $tk_map_y->Button(
	-text => "+", 
	-command => sub {$y_size++, change_canvas_size()}
    )->pack( 
	-side => "left"
    );
    $tk_map_y->Button(
	-text => "-", 
	-command => sub {$y_size--, change_canvas_size()}
    )->pack(
	-side => "left"
    );
    $tk_nav->Button(
	-text => "Save Image",
	-command => \&save_image
    )->pack(
	-side => "top",
	-anchor => "w"
    );
    $tk_nav->Button(
	-text => "Print",
	-command => \&print_image
    )->pack(
	-side => "top",
	-anchor => "w"
    );

    # The frame for the lat/log goto button
    my $tk_lat_long = $tk_nav->Frame(
    )->pack( 
	-side => "top",
	-expand => 1,
	-fill => "x"
    );

    $tk_lat_long->Label(
	-text => "Latitude:"
    )->pack(
	-side => "left"
    );
    $tk_lat_long->Entry(
	-textvariable => \$goto_lat,
	-width => 10
    )->pack(
	-side => "left"
    );
    $tk_lat_long->Label(
	-text => "Longitude"
    )->pack(
	-side => "left"
    );
    $tk_lat_long->Entry(
	-textvariable => \$goto_long,
	-width => 10
    )->pack(
	-side => "left"
    );

    $tk_lat_long->Button(
	-text => "Goto Lat / Long",
	-command => \&goto_lat_long
    )->pack(
	-side => "left"
    );
    $tk_nav->Button(
	-text => "Goto Location",
	-command => \&goto_loc
    )->pack(
	-side => "top",
	-anchor => "w"
    );
    $tk_nav->Button(
	-text => "Exit",
	-command => sub {exit(0);}
    )->pack(
	-side => "top",
	-anchor => "w"
    );

    $tk_nav->bind('<Destroy>', sub { exit(0);});
    $tk_nav->raise();
}


build_gui();

# Grand Canyon (360320N 1120820W)
set_center_lat_long(360320, -1120820);
set_scale(12);

show_map();
$tk_nav->raise();

MainLoop();
